(* ::Subsubsection:: *)
(*HoughtonCassegrainC package*)

(* ::Text:: *)
(*This impements the Houghton-Cassegrain telescope with corrector at the prime focus*)

(* ::Text:: *)
(*Chapter*)

(* ::Text:: *)
(*This file should be in the following path*)

(* ::Text:: *)
(*FileNameJoin[{$UserBaseDirectory, "Applications/GeometricOptics/HoughtonCassegrainC"}]*)

BeginPackage["GeometricOptics`Packages`HoughtonCassegrainC`", {"GeometricOptics`"}]

Options[HoughtonCassegrainC] = {OutputType->"Report", OutputLevel->"Full"};
(* 	OutputType can be 
					"Report", generates a new report (notebook) for each call to the function
					"Print", prints results in the current position of the evaluating notebook
					"Basic", gives the list of results
					"Values" or any other directive, gives the list of values (no names of variables)
	OutputLevel can be
					"Minimal", gives only output values, those calculated by the function
					"Full", gives the list of input parameters and output values
*)

HoughtonCassegrainC::usage="HoughtonCassegrainC calculates parameters for a Houghton-Cassegrain telescope optical system.";
HoughtonCassegrainCInputPanel::usage = "HoughtonCassegrainCInputPanel gives the GUI panel to work with HoughtonCassegrainC";

$HoughtonCassegrainCInputVariables = {"f1", "ft", "ind", "em", "\[Sigma]", "diam", "\[Theta]", "waves", "dcp", "thick", "di"};

Begin["`Private`"]

HoughtonCassegrainC[{f1_, ft_, ind_, em_, sigma_, diam_, theta_, waves_, dcp_, thick_, di_}, opts___]:= HoughtonCassegrainC[f1, ft, ind, em, sigma, diam, theta, waves, dcp, thick, di];

HoughtonCassegrainC[f1_, ft_, ind_, em_, sigma_, diam_, theta_, waves_, dcp_, thick_, di_, OptionsPattern[]] := 
Quiet@Module[{fact, \[Alpha], M, \[Beta], \[Gamma], R, K0, c11, c12, c14, eq1, eq2, eq3, s1, inputs, outputs, inPanel, outPanel, title},
 		If[ArgumentsQ["HoughtonCassegrainC", {$HoughtonCassegrainCInputVariables, {f1, ft, ind, em, sigma, diam, theta, waves, dcp, thick, di}}],
			AppendTo[$ExamplesStack, 
					 <|"PackageID"->11, "PackageName"->"HoughtonCassegrainC", 
						"Arguments"-><|"f1"->f1, "ft"->ft, "ind"->ind, "em"->em, "\[Sigma]"->sigma, "diam"->diam, "\[Theta]"->theta, "waves"->waves, "dcp"->dcp, "thick"->thick, "di"->di|>|>];
			$ExamplesStack = DeleteExampleDuplicates[$ExamplesStack];
		
			(*First step: a Dall-Kirkham combination without corrector and with a focal fact ft a back focal sigma em*)
			fact = 0.8;
			\[Alpha] = (sigma em)/f1;
			M = (fact ft)/f1;
			\[Beta] = (M - \[Alpha])/(M + 1);
			\[Gamma] = (2 M (1 + \[Alpha]))/(1 - M^2);
			R2 = \[Gamma] f1;
			K0 = -(( 1 + M^3 + \[Alpha] - M^2 (1 + \[Alpha]))/M^3);
			(* Second step: the Dall-Kirkham combination and a corrector with two equal radii and thickness equal to zero; three unlnowns c11, c12, c14 *)
			TotalAberrations[{-2 f1, \[Gamma] f1, 1/c11, 1/c12, -1/c11, 1/c14}, {-\[Beta] f1, \[Beta] f1 - dcp, 0, 0, 0}, ind, {K0, 0, 0, 0, 0, 0}, diam/2, 0, 0, -Infinity, y, theta, waves];
			
			eq1 = Chop[Numerator[Together[GOaberration[[1]]]], 10^(-20)];
			eq2 = Chop[Numerator[Together[GOcoma[[1]]]], 10^(-20)];
			eq3 = Chop[Numerator[Together[(GOdistancegauss[[2, GOn]] - GOdistancegauss[[3, GOn]])]], 10^(-20)];
			
			s1 = Flatten@Cases[ReplaceAll[{c11, c12, c14}, Solve[{eq1 == 0, eq2 == 0, eq3 == 0}, {c11, c12, c14}, Reals]], _?(Norm[#] < 1 &)];
						
			(* third step: the thickness of corrector is taken into account and the radii are different; further, the radius of primary is unknown; 6 unknowns K1,c0,c11,c12,c13,c14. 
				These unknowns are determined imposing that spherical aberration, coma, astigmatism vanish and total focal and back distance are given *)
			TotalAberrations[{1/c0, \[Gamma] f1, 1/c21, 1/c22, 1/c23, 1/c24}, {-\[Beta] f1, \[Beta] f1 - dcp, thick[[1]], di, thick[[2]]}, 
								ind, {K1, 0, 0, 0, 0, 0}, diam/2, 0, 0, -Infinity, y, theta, waves];
			{c0, K1, c21, c22, c23, c24} = ReplaceAll[{c0, K1, c21, c22, c23, c24},
														FindRoot[{GOfocalLength[[1]] == ft, GOaberration[[1]] == 0, GOcoma[[1]] == 0, GOdistancegauss[[2, GOn]] - GOdistancegauss[[3, GOn]] == 0, 
																	GOastigmatism[[1]] == 10^(-5), GOdistancegauss[[1, GOn]] == em + dcp - (thick[[1]] + di + thick[[2]])}, 
																	{c0, -1/(2 f1)}, {K1, K0}, {c21, s1[[1]]}, {c22, s1[[2]]}, {c23, -s1[[1]]}, {c24, s1[[3]]}, 
																AccuracyGoal -> Infinity, PrecisionGoal -> 10, MaxIterations -> 1000]];
			TotalAberrations[{1/c0, \[Gamma] f1, 1/c21, 1/c22, 1/c23, 1/c24}, {-\[Beta] f1, \[Beta] f1 - dcp, thick[[1]], di, thick[[2]]}, ind, {K1, 0, 0, 0, 0, 0}, diam/2, 0, 0, -Infinity, y, theta, waves];
			
			(* We are now in condition to evaluate all the caracteristics of the combination, the higher-order aberrations and the radii of corrector that eliminate the third-order aberrations *)
						  	  
			outputs = {{"Radius of primary mirror", 1/c0},
					   {"Radius of secondary mirror", \[Gamma] f1},
					   {"Distance between the secondary and primary", -\[Beta] f1},
					   {"Distance between secondary mirror and the first surface of corrector", \[Beta] f1 - dcp},
					   {"Total focal lenght", GOfocalLength[[1]]},
					   {"Conic constant", K1},
					   {"Radii of curvature of corrector", Row[Riffle[{1/c21, 1/c22, 1/c23, 1/c24},", "]]},
					   {"Third-order spherical aberration", GOaberration[[1]]},
					   {"Third-order coma", GOcoma[[1]]},
					   {"Axial chromatism", (GOdistancegauss[[2, GOn]] - GOdistancegauss[[3, GOn]])}};

			(* defines the two panels, input parameters and output values *)
			inPanel = Grid[{{"Focal length of the primary mirror" , "f1", f1},
							{"Total focal length", "ft", ft},
							{"List of refractive indices", "ind", Grid[ind]},
							{"Back distance", "em", em},
							{"Fraction of em in the first step of the project", "\[Sigma]", sigma},
							{"Diameter of the primary mirror", "diam", diam},
							{"Field angle in degrees", "\[Theta]", theta},
							{"Wavelengths of the choose refractive indices", "waves", waves},
							{"Distance between the first surface of the corrector and the primary", "dcp", dcp},
							{"Thickness of lenses" , "thick", thick},
							{"Distance between lenses", "di", di}},
							Alignment -> {{Left, Left, Right}, Center}, 
							Spacings -> {2, 1}, 
							Dividers -> Center, 
							FrameStyle -> LightGray,
							BaseStyle->{"InputParameterBottom"}];
  
			outPanel = Grid[outputs, 
							Alignment -> {{Left, Right}, Center}, 
							Spacings -> {2, 1}, 
							Dividers -> Center, 
							FrameStyle -> LightGray,
							BaseStyle->{"OutputValueBottom"}];

			
			title = $GeometricOpticsPackagesList[SelectFirst[#PackageName == "HoughtonCassegrainC" &], "Description"];
			
			(* generates the type of output required *)
			Switch[OptionValue[OutputType],
					"Report",
					GenerateDocument[TemplateApply[$ReportTemplate, 
										Join[<|	"title" -> title,																	
												"date" -> DateString[], 
												"function" -> "HoughtonCassegrainC", 
												"outputlevel" -> OptionValue[OutputLevel],
												"inPanel" -> inPanel, 
												"outPanel" -> outPanel |>]]];,

					"Print",
					CellPrint[TextCell[TemplateApply[$PrintTemplate, 
											Join[<|	"title" -> title, 
													"date" -> DateString[], 
													"function" -> "HoughtonCassegrainC", 
													"outputlevel" -> OptionValue[OutputLevel],
													"inPanel" -> inPanel, 
													"outPanel" -> outPanel |>]], "Text"]];,
					"Basic",
					CellPrint[TextCell[
					TemplateApply[$BasicTemplate, 
									Join[<| "outputlevel" -> OptionValue[OutputLevel],
											"inputs" -> {{"f1", f1},
														 {"ft", ft},
														 {"ind", ind},
														 {"", },
														 {"em", em},
														 {"\[Sigma]", sigma},
														 {"diam", diam},
														 {"\[Theta]", theta},
														 {"waves", waves},
														 {"dcp", dcp},
														 {"thick", thick},
														 {"di", di}														 
														 },
											"outputs" -> outputs |>]], "Output"]],
					_,
					CellPrint[TextCell[DeleteCases[outputs[[All, 2]], Alternatives["", Style[___]]], "Output"]]],

							
			(* Arguments are not correct *)
			MessageDialog["HoughtonCassegrainC not executed, the number or the type of the arguments may be incorrect.", WindowTitle->"Warning: example not generated"];]];

HoughtonCassegrainCInputPanel[]:=
DynamicModule[{outputtype = "Report", package = "HoughtonCassegrainC"},
	examplesAll = Join[	Cases[Values[$ExamplesStack], {_, package, arguments_} :> arguments], 
						Cases[Normal@Values[$ExamplesArchive], {_, _, package, arguments_, _} :> arguments]];
    examples = Map[Framed[Grid[Transpose[KeyValueMap[List, Association[#]]], Alignment -> Center, Spacings -> {1, 1}, Dividers -> Center], FrameStyle -> LightGray] &, examplesAll];
	example = "Browse...";
	Panel[Column[{	DynamicWrapper[Style[NameFromPackageName[package] <> " Input Panel", "Subsection"], 
									If[NumericQ[example], {f1, ft, ind, em, sigma, diam, theta, waves, dcp, thick, di} = ReplaceAll[$HoughtonCassegrainCInputVariables, examplesAll[[example]]]]],
					Style["Insert values for each argument, then use Evaluate to run HoughtonCassegrainC function", "Text"],
					Grid[{{"Focal length of the primary mirror" , "f1", Tooltip[InputField[Dynamic[f1], Alignment -> Center], "Insert the focal length of the primary mirror"]},
						  {"Total focal length", "ft", Tooltip[InputField[Dynamic[ft], Alignment -> Center], "Insert the total focal length"]},
						  {"List of refractive indices", "ind", Tooltip[InputField[Dynamic[ind], Alignment -> Center, FieldSize->30], "Insert the list of refractive indices"]},
						  {"Back distance", "em", Tooltip[InputField[Dynamic[em], Alignment -> Center], "Insert the back distance"]},
						  {"Fraction of em in the first step of the project", "\[Sigma]", Tooltip[InputField[Dynamic[sigma], Alignment -> Center], "Insert the fraction of em in the first step of the project"]},
						  {"Diameter of the primary mirror", "diam", Tooltip[InputField[Dynamic[diam], Alignment -> Center], "Input the value of diam"]},
						  {"Field angle in degrees", "\[Theta]", Tooltip[InputField[Dynamic[theta], Alignment -> Center], "Insert the value of \[Theta]"]},
						  {"Wavelengths of the choose refractive indices", "waves", Tooltip[InputField[Dynamic[waves], Alignment -> Center], "Input the list of wavelengths"]},
						  {"Distance between the first surface of the corrector and the primary", "dcp", Tooltip[InputField[Dynamic[dcp], Alignment -> Center], "Input the distance between the first surface of the corrector and the primary"]},
						  {"Thickness of lenses" , "thick", Tooltip[InputField[Dynamic[thick], Alignment -> Center], "Insert the list of thickness of lenses"]},
						  {"Distance between lenses", "di", Tooltip[InputField[Dynamic[di], Alignment -> Center], "Input the distance between lenses"]}
						  },
						  
						Spacings -> {1, 0},
						Alignment -> {{Left, Left, Right}, Center}, 
						Dividers -> Center, 
						FrameStyle -> LightGray],
					OpenerView[{"Load an example from the archives (current session and saved DB)",
								Row[{Dynamic@PopupMenu[Dynamic[example], Thread[Rule[Range[Length[examples]], examples]], If[examples === {}, "No example saved", "Browse..."], FrameMargins -> 3, Alignment -> Center],
									 Spacer[5],
									 Button["Update list", (examplesAll = Join[	Cases[Values[$ExamplesStack], {_, package, arguments_} :> arguments], 
																				Cases[Normal@Values[$ExamplesArchive], {_, _, package, arguments_, _} :> arguments]];
															examples = Map[Framed[Grid[Transpose[KeyValueMap[List, Association[#]]], 
																						Alignment -> Center, 
																						Spacings -> {1, 1}, 
																						Dividers -> Center], 
																						FrameStyle -> LightGray]&, examplesAll];
															example = "Browse..."), Method -> "Queued"]}]}, 
								Alignment -> Center, 
								Spacings -> 1.5],
					Row[{"Define the type of output to generate", 
						 Spacer[5],
						 RadioButtonBar[Dynamic[outputtype], {"Report" -> Tooltip["Report", "Generates a new notebook reporting a summary of the calculation"], 
															  "Print" -> Tooltip["Print", "Print the table of the calculation done inside the current notebook"], 
															  "Basic" -> Tooltip["Basic", "Generate a list of computed output with label"], 
															  "Values" -> Tooltip["Values", "Return only the list of output values"]}]}],
					Row[{Button["Evaluate", ToExpression[package][Apply[Sequence, {f1, ft, ind, em, sigma, diam, theta, waves, dcp, thick, di}], OutputType -> outputtype], Method -> "Queued"],
						 Button["Clear all", Map[Clear, Unevaluated[{f1, ft, ind, em, sigma, diam, theta, waves, dcp, thick, di}]]]}]}, 
				Spacings -> 2, 
				Alignment -> Center],
	BaseStyle -> {InputFieldBoxOptions -> {FieldSize -> {15, 1}}}]];
			
  
  End[]
  EndPackage[]